<?php
/**
 * Simple CSV Importer Module
 * 
 * Allows admin to bulk import data for Schedule, Syllabus, Calendar, and Asatidz via CSV.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// 1. Register Admin Menu
function pondok_register_importer_page() {
    add_submenu_page(
        'pondok-theme-settings', // Parent slug (Fixed: matched with Theme Options)
        'Import Data',          // Page title
        'Import Data',          // Menu title
        'manage_options',       // Capability
        'pondok-csv-importer',  // Menu slug
        'pondok_render_importer_page' // Callback function
    );
}
add_action('admin_menu', 'pondok_register_importer_page');

// 2. Render Page UI
function pondok_render_importer_page() {
    // Check capability
    if (!current_user_can('manage_options')) {
        return;
    }

    // LIST OF AVAILABLE IMPORT TYPES
    $import_types = array(
        'jadwal' => array(
            'label' => 'Jadwal Pelajaran',
            'option_key' => 'pondok_jadwal_list',
            'template' => 'template_jadwal.xls',
            'desc' => 'Kolom wajib: Hari, Jam_Mulai, Jam_Selesai, Kelas, Mata_Pelajaran, Pengajar, Tingkatan. <strong>(Download XLS, Edit, Save As CSV)</strong>'
        ),
        'silabus' => array(
            'label' => 'Silabus Kurikulum',
            'option_key' => 'pondok_silabus_list',
            'template' => 'template_silabus.xls',
            'desc' => 'Kolom wajib: Mata_Pelajaran, Kode_Mapel, Kategori, Deskripsi, Tingkatan. <strong>(Download XLS, Edit, Save As CSV)</strong>'
        ),
        'agenda' => array(
            'label' => 'Agenda Kalender',
            'option_key' => 'pondok_agenda_list',
            'template' => 'template_agenda.xls',
            'desc' => 'Kolom wajib: Tanggal (YYYY-MM-DD), Judul_Kegiatan, Kategori, Waktu_Mulai. <strong>(Download XLS, Edit, Save As CSV)</strong>'
        ),
        'asatidz' => array(
            'label' => 'Data Asatidz',
            'option_key' => 'pondok_asatidz_list',
            'template' => 'template_asatidz.xls',
            'desc' => 'Kolom wajib: Nama, Jabatan, Bidang, Bio_Singkat. <strong>(Download XLS, Edit, Save As CSV)</strong>'
        ),
    );

    // MESSAGE HANDLER
    if (isset($_GET['status']) && $_GET['status'] == 'success') {
        $count = isset($_GET['count']) ? intval($_GET['count']) : 0;
        echo '<div class="notice notice-success is-dismissible"><p>Berhasil mengimpor <strong>' . $count . '</strong> baris data.</p></div>';
    }
    if (isset($_GET['status']) && $_GET['status'] == 'error') {
        $msg = isset($_GET['msg']) ? sanitize_text_field($_GET['msg']) : 'Terjadi kesalahan.';
        echo '<div class="notice notice-error is-dismissible"><p>Gagal: ' . $msg . '</p></div>';
    }

    ?>
    <div class="wrap">
        <h1>Import Data Massal (CSV)</h1>
        <p>Gunakan fitur ini untuk mengisi data website secara cepat menggunakan file Excel/CSV.</p>
        <hr class="wp-header-end">

        <div style="display: flex; gap: 20px; flex-wrap: wrap; margin-top: 20px;">
            
            <!-- LEFT COLUMN: UPLOAD FORM -->
            <div class="card" style="padding: 20px; flex: 1; min-width: 300px; max-width: 500px;">
                <h2>Upload File CSV</h2>
                <form method="post" enctype="multipart/form-data" action="<?php echo admin_url('admin-post.php'); ?>">
                    <input type="hidden" name="action" value="pondok_process_csv_import">
                    <?php wp_nonce_field('pondok_csv_import_verify', 'pondok_nonce'); ?>

                    <p>
                        <label for="import_type"><strong>1. Pilih Jenis Data:</strong></label><br>
                        <select name="import_type" id="import_type" style="width: 100%; margin-top: 5px;" required onchange="updateTemplateLink(this.value)">
                            <option value="">-- Pilih Jenis --</option>
                            <?php foreach ($import_types as $key => $info) : ?>
                                <option value="<?php echo esc_attr($key); ?>"><?php echo esc_html($info['label']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </p>

                    <p>
                        <label for="import_mode"><strong>2. Mode Import:</strong></label><br>
                        <label><input type="radio" name="import_mode" value="append" checked> Tambahkan ke data yang ada (Append)</label><br>
                        <label><input type="radio" name="import_mode" value="replace"> Hapus data lama & Ganti Baru (Replace)</label>
                    </p>

                    <p>
                        <label for="csv_file"><strong>3. Pilih File (.csv):</strong></label><br>
                        <input type="file" name="csv_file" id="csv_file" accept=".csv" required style="margin-top: 5px;">
                    </p>

                    <p class="submit">
                        <button type="submit" class="button button-primary button-large">Mulai Proses Import</button>
                    </p>
                </form>
            </div>

            <!-- RIGHT COLUMN: GUIDE & TEMPLATES -->
            <div class="card" style="padding: 20px; flex: 1; min-width: 300px;">
                <h2>Panduan & Template</h2>
                <p>Agar proses import berhasil, pastikan file CSV Anda mengikuti format yang benar. Silakan unduh template di bawah ini:</p>

                <div id="template-info-box" style="background: #f0f6fc; border: 1px solid #d0d7de; padding: 15px; border-radius: 6px;">
                    <p style="margin-top:0;"><strong>Silakan pilih "Jenis Data" di samping kiri untuk melihat panduan spesifik.</strong></p>
                </div>

                <div style="margin-top: 20px;">
                    <h3>Daftar Template Tersedia:</h3>
                    <ul>
                    <?php 
                    $template_url_base = get_template_directory_uri() . '/assets/templates/';
                    foreach ($import_types as $key => $info) : ?>
                        <li>
                            <a href="<?php echo $template_url_base . $info['template']; ?>" download class="button button-secondary" style="margin-bottom: 5px;">
                                <span class="dashicons dashicons-download" style="line-height: 28px;"></span> 
                                Download Template <?php echo $info['label']; ?>
                            </a>
                        </li>
                    <?php endforeach; ?>
                    </ul>
                </div>
            </div>

        </div>
    </div>

    <!-- JS for Dynamic Guide -->
    <script>
    var typeInfo = <?php echo json_encode($import_types); ?>;
    
    function updateTemplateLink(val) {
        var box = document.getElementById('template-info-box');
        if (!val || !typeInfo[val]) {
            box.innerHTML = '<p style="margin-top:0;"><strong>Silakan pilih "Jenis Data" di samping kiri untuk melihat panduan spesifik.</strong></p>';
            return;
        }
        
        var info = typeInfo[val];
        var tplUrl = '<?php echo get_template_directory_uri() . '/assets/templates/'; ?>' + info['template'];

        var html = '<h3 style="margin-top:0;">Format: ' + info.label + '</h3>';
        html += '<p>' + info.desc + '</p>';
        html += '<a href="' + tplUrl + '" class="button button-primary">Download Template Ini</a>';
        
        box.innerHTML = html;
    }
    </script>
    <?php
}

// 3. Handle Form Submission
add_action('admin_post_pondok_process_csv_import', 'pondok_process_csv_import');

function pondok_process_csv_import() {
    // 0. PHP Fix for Macintosh/Windows Line Endings
    ini_set('auto_detect_line_endings', true);

    // 1. Check Permissions & Nonce
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }
    check_admin_referer('pondok_csv_import_verify', 'pondok_nonce');

    // 2. Validate File
    if (empty($_FILES['csv_file']['tmp_name'])) {
        wp_redirect(admin_url('admin.php?page=pondok-csv-importer&status=error&msg=File kosong'));
        exit;
    }
    
    $file = $_FILES['csv_file']['tmp_name'];
    $type = sanitize_text_field($_POST['import_type']);
    $mode = sanitize_text_field($_POST['import_mode']);

    // Map Types to Option Keys
    // FIX: Key must match exactly with settings files
    $map = array(
        'jadwal' => 'pondok_jadwal_list',  // CORRECTED from pondok_jadwal_kbm_list
        'silabus' => 'pondok_silabus_list',
        'agenda' => 'pondok_agenda_list',
        'asatidz' => 'pondok_asatidz_list'
    );

    if (!array_key_exists($type, $map)) {
        wp_redirect(admin_url('admin.php?page=pondok-csv-importer&status=error&msg=Tipe data tidak valid'));
        exit;
    }
    $target_option = $map[$type];

    // 3. Process CSV
    // Read entire file into array
    $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if ($lines === false || empty($lines)) {
        wp_redirect(admin_url('admin.php?page=pondok-csv-importer&status=error&msg=Gagal membaca file atau file kosong'));
        exit;
    }

    // Process Header
    $header_line = array_shift($lines);
    // Remove BOM
    $header_line = preg_replace('/\x{FEFF}/u', '', $header_line);
    
    // AUTO DETECT DELIMITER
    $delimiter = ',';
    if (substr_count($header_line, ';') > substr_count($header_line, ',')) {
        $delimiter = ';';
    }

    $header = str_getcsv($header_line, $delimiter);
    
    // Normalize Header
    $header = array_map(function($h) {
        return strtolower(trim($h)); // e.g. "Mata_Pelajaran" -> "mata_pelajaran"
    }, $header);

    // DEBUG LOG
    error_log('CSV IMPORT START --------------------------------');
    error_log('Detected Delimiter: ' . $delimiter);
    error_log('Header Parsed: ' . print_r($header, true));

    $new_data = array();
    $focus_level = function_exists('pondok_get_option') ? pondok_get_option('school_level_focus', '') : get_option('pondok_school_level_focus', '');
    error_log('Active Focus Level: ' . $focus_level);

    $process_count = 0;

    foreach ($lines as $line_raw) {
        $row = str_getcsv($line_raw, $delimiter);
        $process_count++;
        
        // Debug first row data
        if ($process_count == 1) {
             error_log('First Row Raw: ' . print_r($row, true));
        }

        // Skip if column count mismatch
        if (count($row) !== count($header)) {
            error_log("Row #{$process_count} Skipped: Column count mismatch. Header: " . count($header) . ", Row: " . count($row));
            continue; 
        }

        $item_raw = array_combine($header, $row);
        $item_clean = array();

        // 3.1 Type Specific Mapping & Validation
        if ($type == 'jadwal') {
            // Focus Level Check (Disabled temporarily to ensure import works)
            $row_level = isset($item_raw['tingkatan']) ? trim($item_raw['tingkatan']) : '';
            /*
            if (!empty($focus_level) && strcasecmp($row_level, $focus_level) !== 0) {
                 continue; 
            }
            */

            $item_clean = array(
                'hari' => sanitize_text_field(isset($item_raw['hari']) ? $item_raw['hari'] : ''),
                'jam_mulai' => sanitize_text_field(isset($item_raw['jam_mulai']) ? $item_raw['jam_mulai'] : ''),
                'jam_selesai' => sanitize_text_field(isset($item_raw['jam_selesai']) ? $item_raw['jam_selesai'] : ''),
                'mapel' => sanitize_text_field(isset($item_raw['mata_pelajaran']) ? $item_raw['mata_pelajaran'] : ''),
                'kelas' => sanitize_text_field(isset($item_raw['kelas']) ? $item_raw['kelas'] : ''),
                'pengajar' => sanitize_text_field(isset($item_raw['pengajar']) ? $item_raw['pengajar'] : ''),
                'ruang' => sanitize_text_field(isset($item_raw['ruang']) ? $item_raw['ruang'] : 'R.Umum'), 
                'status' => sanitize_text_field(isset($item_raw['status']) ? $item_raw['status'] : 'Aktif'),
                'tingkat' => $row_level,
            );

        } elseif ($type == 'silabus') {
            // Focus Level Check (Disabled temporarily)
            $row_level = isset($item_raw['tingkatan']) ? trim($item_raw['tingkatan']) : '';
            /*
            if (!empty($focus_level) && strcasecmp($row_level, $focus_level) !== 0) {
                continue; 
            }
            */

            $item_clean = array(
                'judul' => sanitize_text_field(isset($item_raw['mata_pelajaran']) ? $item_raw['mata_pelajaran'] : ''),
                'kode' => sanitize_text_field(isset($item_raw['kode_mapel']) ? $item_raw['kode_mapel'] : ''),
                'kategori' => sanitize_text_field(isset($item_raw['kategori']) ? $item_raw['kategori'] : ''),
                'deskripsi' => sanitize_textarea_field(isset($item_raw['deskripsi']) ? $item_raw['deskripsi'] : ''),
                'level' => $row_level,
                'file_url' => '', // CSV can't handle files
                'referensi' => '',
                'topik' => ''
            );

        } elseif ($type == 'agenda') {
            $item_clean = array(
                'date' => sanitize_text_field(isset($item_raw['tanggal']) ? $item_raw['tanggal'] : ''),
                'title' => sanitize_text_field(isset($item_raw['judul_kegiatan']) ? $item_raw['judul_kegiatan'] : ''),
                'category' => sanitize_text_field(isset($item_raw['kategori']) ? $item_raw['kategori'] : 'Akademik'),
                'time' => sanitize_text_field(isset($item_raw['waktu_mulai']) ? $item_raw['waktu_mulai'] : ''),
            );

        } elseif ($type == 'asatidz') {
            $item_clean = array(
                'nama' => sanitize_text_field(isset($item_raw['nama']) ? $item_raw['nama'] : ''),
                'jabatan' => sanitize_text_field(isset($item_raw['jabatan']) ? $item_raw['jabatan'] : ''),
                'bidang' => sanitize_text_field(isset($item_raw['bidang']) ? $item_raw['bidang'] : ''),
                'bio' => sanitize_textarea_field(isset($item_raw['bio_singkat']) ? $item_raw['bio_singkat'] : ''),
                'foto' => '', 
            );
        }

        if (!empty($item_clean)) {
            $new_data[] = $item_clean;
        }
    }

    // 4. Save to Database
    if ($mode == 'replace') {
        $final_data = $new_data;
    } else {
        // Append
        $existing_data = get_option($target_option, array());
        if(!is_array($existing_data)) $existing_data = array();
        $final_data = array_merge($existing_data, $new_data);
    }
    
    // Auto sort for calendar
    if ($type == 'agenda') {
        usort($final_data, function($a, $b) { return strtotime($a['date']) - strtotime($b['date']); });
    }

    update_option($target_option, $final_data);

    // 5. Redirect Success
    wp_redirect(admin_url('admin.php?page=pondok-csv-importer&status=success&count=' . count($new_data)));
    exit;
}
